<?php
namespace TondaCore\CPT\Shortcodes\ProductBanner;

use TondaCore\Lib;

class ProductBanner implements Lib\ShortcodeInterface {
	private $base;
	
	function __construct() {
		$this->base = 'qodef_product_banner';
		
		add_action('vc_before_init', array($this,'vcMap'));
		
		//Product id filter
		add_filter( 'vc_autocomplete_qodef_product_banner_product_id_callback', array( &$this, 'productIdAutocompleteSuggester', ), 10, 1 );
		
		//Product id render
		add_filter( 'vc_autocomplete_qodef_product_banner_product_id_render', array( &$this, 'productIdAutocompleteRender', ), 10, 1 );
	}
	
	public function getBase() {
		return $this->base;
	}
	
	public function vcMap() {
		if(function_exists('vc_map')) {
			vc_map(
				array(
					'name'                      => esc_html__( 'Product Banner', 'tonda' ),
					'base'                      => $this->getBase(),
					'category'                  => esc_html__( 'by TONDA', 'tonda' ),
					'icon'                      => 'icon-wpb-product-banner extended-custom-icon',
					'allowed_container_element' => 'vc_row',
					'params'                    => array(
						array(
							'type'        => 'autocomplete',
							'param_name'  => 'product_id',
							'heading'     => esc_html__( 'Selected Product', 'tonda' ),
							'settings'    => array(
								'sortable'      => true,
								'unique_values' => true
							),
							'admin_label' => true,
							'save_always' => true,
							'description' => esc_html__( 'If you left this field empty then product ID will be of the current page', 'tonda' )
						),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'title_tag',
							'heading'     => esc_html__( 'Title Tag', 'tonda' ),
							'value'       => array_flip( tonda_select_get_title_tag( true, array( 'p' => 'p' ) ) ),
							'save_always' => true,
							'description' => esc_html__( 'Set title tag for product title element', 'tonda' )
						),
						array(
							'type'        => 'textfield',
							'param_name'  => 'title_break_words',
							'heading'     => esc_html__( 'Position of Line Breaks', 'tonda' ),
							'description' => esc_html__( 'Enter the positions of the words after which you would like to create a line break (e.g. if you would like the line break after first, third, and fourth word , you would enter "1,3,4")', 'tonda' )
						),
						array(
							'type'       => 'dropdown',
							'param_name' => 'featured_image_size',
							'heading'    => esc_html__( 'Featured Image Proportions', 'tonda' ),
							'value'      => array(
								esc_html__( 'Default', 'tonda' )        => '',
								esc_html__( 'Original', 'tonda' )       => 'full',
								esc_html__( 'Square', 'tonda' )         => 'tonda_select_image_square',
								esc_html__( 'Landscape', 'tonda' )      => 'tonda_select_image_landscape',
								esc_html__( 'Portrait', 'tonda' )       => 'tonda_select_image_portrait',
								esc_html__( 'Custom', 'tonda')          => 'custom'
							),
							'save_always' => true
						),
						array(
				            'type'        => 'textfield',
				            'param_name'  => 'featured_custom_image_width',
				            'heading'     => esc_html__('Custom Featured Image Width', 'tonda'),
				            'description' => esc_html__('Enter Featured image width in px', 'tonda'),
				            'dependency'  => array('element' => 'featured_image_size', 'value' => 'custom')
				        ),
				        array(
				            'type'        => 'textfield',
				            'param_name'  => 'featured_custom_image_height',
				            'heading'     => esc_html__('Custom Featured Image Height', 'tonda'),
				            'description' => esc_html__('Enter Featured image height in px', 'tonda'),
				            'dependency'  => array('element' => 'featured_image_size', 'value' => 'custom')
				        ),
						array(
							'type'        => 'dropdown',
							'param_name'  => 'skin',
							'heading'     => esc_html__( 'Skin', 'tonda' ),
							'value'       => array(
								esc_html__( 'Default', 'tonda' ) => '',
								esc_html__( 'Dark', 'tonda' )    => 'dark'
							),
							'save_always' => true
						),
					)
				)
			);
		}
    }
	
    public function render($atts, $content = null) {
        $args = array(
	        'product_id'                   => '',
	        'title_tag'                    => 'h2',
	        'title_break_words'            => '',
	        'featured_image_size'          => 'full',
	        'featured_custom_image_width'  => '',
	        'featured_custom_image_height' => '',
	        'skin'                         => ''
        );
		$params = shortcode_atts($args, $atts);

	    $params['product_id']          = ! empty( $params['product_id'] ) ? $params['product_id'] : get_the_ID();
	    $params['title_tag']           = ! empty( $params['title_tag'] ) ? $params['title_tag'] : $args['title_tag'];
	    $params['featured_image_size'] = ! empty( $params['featured_image_size'] ) ? $params['featured_image_size'] : $args['featured_image_size'];
	    $params['skin']                = ! empty( $params['skin'] ) &&
	                                     in_array( $params['skin'], array( 'dark' ) ) ? $params['skin'] : $args['skin'];

	    $classes = array('qodef-product-banner');
	    if (! empty( $params['skin'] )) {
	    	$classes[] = 'qodef-product-banner-' . $params['skin'];
	    }

	    $product = wc_get_product( $params['product_id'] );

	    $product_params = array(
		    'product_id'                   => $params['product_id'],
		    'categories'                   => wp_get_post_terms( $params['product_id'], 'product_cat' ),
		    'title'                        => get_the_title( $params['product_id'] ),
		    'title_break_words'            => $params['title_break_words'],
		    'title_tag'                    => $params['title_tag'],
		    'featured_image_size'          => $params['featured_image_size'],
		    'featured_custom_image_width'  => tonda_select_filter_px( $params['featured_custom_image_width'] ),
		    'featured_custom_image_height' => tonda_select_filter_px( $params['featured_custom_image_height'] ),
		    'price'                        => ! empty( $product ) ? $product->get_price_html() : '',
		    'classes'                      => $classes
	    );

	    $product_params['title'] = $this->getModifiedTitle( $product_params );

		$html = tonda_select_get_woo_shortcode_module_template_part( 'templates/product-banner-template', 'product-banner', '', $product_params );

        return $html;
	}

	private function getModifiedTitle( $params ) {
		$title             = $params['title'];
		$title_break_words = str_replace( ' ', '', $params['title_break_words'] );

		if ( ! empty( $title ) ) {
			$split_title = explode( ' ', $title );
			$break_words = explode( ',', $title_break_words );
			$end_string  = count( $split_title );

			if ( ! empty( $title_break_words ) ) {
				foreach ( $break_words as $value ) {
					if ( ! empty( $split_title[ $value - 1 ] ) ) {
						$split_title[ $value - 1 ] = $split_title[ $value - 1 ] . '<br />';
					}
				}
			}

			$title = implode( ' ', $split_title );
		}

		return $title;
	}

	/**
	 * Filter product by ID or Title
	 *
	 * @param $query
	 *
	 * @return array
	 */
	public function productIdAutocompleteSuggester( $query ) {
		global $wpdb;
		$product_id = (int) $query;
		$post_meta_infos = $wpdb->get_results( $wpdb->prepare( "SELECT ID AS id, post_title AS title
					FROM {$wpdb->posts} 
					WHERE post_type = 'product' AND ( ID = '%d' OR post_title LIKE '%%%s%%' )", $product_id > 0 ? $product_id : - 1, stripslashes( $query ), stripslashes( $query ) ), ARRAY_A );

		$results = array();
		if ( is_array( $post_meta_infos ) && ! empty( $post_meta_infos ) ) {
			foreach ( $post_meta_infos as $value ) {
				$data = array();
				$data['value'] = $value['id'];
				$data['label'] = esc_html__( 'Id', 'tonda' ) . ': ' . $value['id'] . ( ( strlen( $value['title'] ) > 0 ) ? ' - ' . esc_html__( 'Title', 'tonda' ) . ': ' . $value['title'] : '' );
				$results[] = $data;
			}
		}

		return $results;

	}

	/**
	 * Find product by id
	 * @since 4.4
	 *
	 * @param $query
	 *
	 * @return bool|array
	 */
	public function productIdAutocompleteRender( $query ) {
		$query = trim( $query['value'] ); // get value from requested
		if ( ! empty( $query ) ) {
			
			$product = get_post( (int) $query );
			if ( ! is_wp_error( $product ) ) {
				
				$product_id = $product->ID;
				$product_title = $product->post_title;
				
				$product_title_display = '';
				if ( ! empty( $product_title ) ) {
					$product_title_display = ' - ' . esc_html__( 'Title', 'tonda' ) . ': ' . $product_title;
				}
				
				$product_id_display = esc_html__( 'Id', 'tonda' ) . ': ' . $product_id;

				$data          = array();
				$data['value'] = $product_id;
				$data['label'] = $product_id_display . $product_title_display;

				return ! empty( $data ) ? $data : false;
			}

			return false;
		}

		return false;
	}
}